from __future__ import absolute_import, division, print_function

import iotbx.pdb
import mmtbx.model
import libtbx.load_env
from libtbx.test_utils import show_diff, assert_lines_in_text
from six.moves import cStringIO as StringIO

pdb_str1 = """\
ATOM      1  N   GLY A   1      12.928   4.612   6.102  1.00 16.77           N
ATOM      2  CA  GLY A   1      12.885   4.207   4.651  1.00 16.57           C
ATOM      3  C   GLY A   1      13.922   3.140   4.419  1.00 16.16           C
ATOM      4  O   GLY A   1      14.414   2.521   5.381  1.00 16.78           O
ATOM      5  H1  GLY A   1      11.977   4.525   6.514  0.00 16.77           H
ATOM      6  H2  GLY A   1      13.586   3.992   6.616  0.00 16.77           H
ATOM      7  H3  GLY A   1      13.250   5.598   6.177  0.00 16.77           H
ATOM      8  HA2 GLY A   1      11.900   3.815   4.398  0.00 16.57           H
ATOM      9  HA3 GLY A   1      13.100   5.065   4.014  0.00 16.57           H
ATOM     10  N   ASN A   2      14.281   2.923   3.155  1.00 15.02           N
ATOM     11  CA  ASN A   2      15.415   2.038   2.831  1.00 14.10           C
ATOM     12  C   ASN A   2      16.696   2.537   3.427  1.00 13.13           C
ATOM     13  O   ASN A   2      16.959   3.742   3.426  1.00 11.91           O
ATOM     14  CB  ASN A   2      15.591   1.881   1.341  1.00 15.38           C
ATOM     15  CG  ASN A   2      14.353   1.342   0.692  1.00 14.08           C
ATOM     16  OD1 ASN A   2      13.912   0.227   1.016  1.00 17.46           O
ATOM     17  ND2 ASN A   2      13.733   2.155  -0.169  1.00 11.72           N
ATOM     18  H   ASN A   2      13.820   3.334   2.343  0.00 15.02           H
ATOM     19  HA  ASN A   2      15.193   1.057   3.252  0.00 14.10           H
ATOM     20  HB2 ASN A   2      15.813   2.853   0.900  0.00 15.38           H
ATOM     21  HB3 ASN A   2      16.409   1.188   1.146  0.00 15.38           H
ATOM     22 HD21 ASN A   2      12.885   1.845  -0.643  0.00 11.72           H
ATOM     23 HD22 ASN A   2      14.108   3.086  -0.352  0.00 11.72           H
ATOM     24  N   ASN A   3      17.499   1.590   3.905  1.00 12.26           N
ATOM     25  CA  ASN A   3      18.744   1.904   4.589  1.00 11.74           C
ATOM     26  C   ASN A   3      19.982   1.332   3.895  1.00 11.10           C
ATOM     27  O   ASN A   3      20.065   0.119   3.648  1.00 10.42           O
ATOM     28  CB  ASN A   3      18.678   1.378   6.042  1.00 12.15           C
ATOM     29  CG  ASN A   3      19.931   1.739   6.861  1.00 12.82           C
ATOM     30  OD1 ASN A   3      20.235   2.925   7.072  1.00 15.05           O
ATOM     31  ND2 ASN A   3      20.666   0.715   7.306  1.00 13.48           N
ATOM     32  OXT ASN A   3      20.908   2.077   3.576  1.00 11.10           O
ATOM     33  H   ASN A   3      17.311   0.590   3.832  0.00 12.26           H
ATOM     34  HA  ASN A   3      18.863   2.987   4.586  0.00 11.74           H
ATOM     35  HB2 ASN A   3      17.812   1.815   6.539  0.00 12.15           H
ATOM     36  HB3 ASN A   3      18.588   0.292   6.023  0.00 12.15           H
ATOM     37 HD21 ASN A   3      21.508   0.892   7.854  0.00 13.48           H
ATOM     38 HD22 ASN A   3      20.385  -0.243   7.097  0.00 13.48           H
TER
ATOM     39  N   GLY B   1      12.928  -0.254   6.102  1.00 16.77           N
ATOM     40  CA  GLY B   1      12.885  -0.659   4.651  1.00 16.57           C
ATOM     41  C   GLY B   1      13.922  -1.726   4.419  1.00 16.16           C
ATOM     42  O   GLY B   1      14.414  -2.345   5.381  1.00 16.78           O
ATOM     43  H1  GLY B   1      11.977  -0.341   6.514  0.00 16.77           H
ATOM     44  H2  GLY B   1      13.586  -0.874   6.616  0.00 16.77           H
ATOM     45  H3  GLY B   1      13.250   0.732   6.177  0.00 16.77           H
ATOM     46  HA2 GLY B   1      11.900  -1.051   4.398  0.00 16.57           H
ATOM     47  HA3 GLY B   1      13.100   0.199   4.014  0.00 16.57           H
ATOM     48  N   ASN B   2      14.281  -1.943   3.155  1.00 15.02           N
ATOM     49  CA  ASN B   2      15.415  -2.828   2.831  1.00 14.10           C
ATOM     50  C   ASN B   2      16.696  -2.329   3.427  1.00 13.13           C
ATOM     51  O   ASN B   2      16.959  -1.124   3.426  1.00 11.91           O
ATOM     52  CB  ASN B   2      15.591  -2.985   1.341  1.00 15.38           C
ATOM     53  CG  ASN B   2      14.353  -3.524   0.692  1.00 14.08           C
ATOM     54  OD1 ASN B   2      13.912  -4.639   1.016  1.00 17.46           O
ATOM     55  ND2 ASN B   2      13.733  -2.711  -0.169  1.00 11.72           N
ATOM     56  H   ASN B   2      13.820  -1.532   2.343  0.00 15.02           H
ATOM     57  HA  ASN B   2      15.193  -3.809   3.252  0.00 14.10           H
ATOM     58  HB2 ASN B   2      15.813  -2.013   0.900  0.00 15.38           H
ATOM     59  HB3 ASN B   2      16.409  -3.678   1.146  0.00 15.38           H
ATOM     60 HD21 ASN B   2      12.885  -3.021  -0.643  0.00 11.72           H
ATOM     61 HD22 ASN B   2      14.108  -1.780  -0.352  0.00 11.72           H
ATOM     62  N   ASN B   3      17.499  -3.276   3.905  1.00 12.26           N
ATOM     63  CA  ASN B   3      18.744  -2.962   4.589  1.00 11.74           C
ATOM     64  C   ASN B   3      19.982  -3.534   3.895  1.00 11.10           C
ATOM     65  O   ASN B   3      20.065  -4.747   3.648  1.00 10.42           O
ATOM     66  CB  ASN B   3      18.678  -3.488   6.042  1.00 12.15           C
ATOM     67  CG  ASN B   3      19.931  -3.127   6.861  1.00 12.82           C
ATOM     68  OD1 ASN B   3      20.235  -1.941   7.072  1.00 15.05           O
ATOM     69  ND2 ASN B   3      20.666  -4.151   7.306  1.00 13.48           N
ATOM     70  OXT ASN B   3      20.908  -2.789   3.576  1.00 11.10           O
ATOM     71  H   ASN B   3      17.311  -4.276   3.832  0.00 12.26           H
ATOM     72  HA  ASN B   3      18.863  -1.879   4.586  0.00 11.74           H
ATOM     73  HB2 ASN B   3      17.812  -3.051   6.539  0.00 12.15           H
ATOM     74  HB3 ASN B   3      18.588  -4.574   6.023  0.00 12.15           H
ATOM     75 HD21 ASN B   3      21.508  -3.974   7.854  0.00 13.48           H
ATOM     76 HD22 ASN B   3      20.385  -5.109   7.097  0.00 13.48           H
TER
ATOM     77  N   GLY C   1      12.928   9.478   6.102  1.00 16.77           N
ATOM     78  CA  GLY C   1      12.885   9.073   4.651  1.00 16.57           C
ATOM     79  C   GLY C   1      13.922   8.006   4.419  1.00 16.16           C
ATOM     80  O   GLY C   1      14.414   7.387   5.381  1.00 16.78           O
ATOM     81  H1  GLY C   1      11.977   9.391   6.514  0.00 16.77           H
ATOM     82  H2  GLY C   1      13.586   8.858   6.616  0.00 16.77           H
ATOM     83  H3  GLY C   1      13.250  10.464   6.177  0.00 16.77           H
ATOM     84  HA2 GLY C   1      11.900   8.681   4.398  0.00 16.57           H
ATOM     85  HA3 GLY C   1      13.100   9.931   4.014  0.00 16.57           H
ATOM     86  N   ASN C   2      14.281   7.789   3.155  1.00 15.02           N
ATOM     87  CA  ASN C   2      15.415   6.904   2.831  1.00 14.10           C
ATOM     88  C   ASN C   2      16.696   7.403   3.427  1.00 13.13           C
ATOM     89  O   ASN C   2      16.959   8.608   3.426  1.00 11.91           O
ATOM     90  CB  ASN C   2      15.591   6.747   1.341  1.00 15.38           C
ATOM     91  CG  ASN C   2      14.353   6.208   0.692  1.00 14.08           C
ATOM     92  OD1 ASN C   2      13.912   5.093   1.016  1.00 17.46           O
ATOM     93  ND2 ASN C   2      13.733   7.021  -0.169  1.00 11.72           N
ATOM     94  H   ASN C   2      13.820   8.200   2.343  0.00 15.02           H
ATOM     95  HA  ASN C   2      15.193   5.923   3.252  0.00 14.10           H
ATOM     96  HB2 ASN C   2      15.813   7.719   0.900  0.00 15.38           H
ATOM     97  HB3 ASN C   2      16.409   6.054   1.146  0.00 15.38           H
ATOM     98 HD21 ASN C   2      12.885   6.711  -0.643  0.00 11.72           H
ATOM     99 HD22 ASN C   2      14.108   7.952  -0.352  0.00 11.72           H
ATOM    100  N   ASN C   3      17.499   6.456   3.905  1.00 12.26           N
ATOM    101  CA  ASN C   3      18.744   6.770   4.589  1.00 11.74           C
ATOM    102  C   ASN C   3      19.982   6.198   3.895  1.00 11.10           C
ATOM    103  O   ASN C   3      20.065   4.985   3.648  1.00 10.42           O
ATOM    104  CB  ASN C   3      18.678   6.244   6.042  1.00 12.15           C
ATOM    105  CG  ASN C   3      19.931   6.605   6.861  1.00 12.82           C
ATOM    106  OD1 ASN C   3      20.235   7.791   7.072  1.00 15.05           O
ATOM    107  ND2 ASN C   3      20.666   5.581   7.306  1.00 13.48           N
ATOM    108  OXT ASN C   3      20.908   6.943   3.576  1.00 11.10           O
ATOM    109  H   ASN C   3      17.311   5.456   3.832  0.00 12.26           H
ATOM    110  HA  ASN C   3      18.863   7.853   4.586  0.00 11.74           H
ATOM    111  HB2 ASN C   3      17.812   6.681   6.539  0.00 12.15           H
ATOM    112  HB3 ASN C   3      18.588   5.158   6.023  0.00 12.15           H
ATOM    113 HD21 ASN C   3      21.508   5.758   7.854  0.00 13.48           H
ATOM    114 HD22 ASN C   3      20.385   4.623   7.097  0.00 13.48           H
TER
ATOM    115  N   GLY D   1       9.009   2.179  -6.102  1.00 16.77           N
ATOM    116  CA  GLY D   1       9.052   1.774  -4.651  1.00 16.57           C
ATOM    117  C   GLY D   1       8.015   0.707  -4.419  1.00 16.16           C
ATOM    118  O   GLY D   1       7.523   0.088  -5.381  1.00 16.78           O
ATOM    119  H1  GLY D   1       9.960   2.092  -6.514  0.00 16.77           H
ATOM    120  H2  GLY D   1       8.351   1.559  -6.616  0.00 16.77           H
ATOM    121  H3  GLY D   1       8.687   3.165  -6.177  0.00 16.77           H
ATOM    122  HA2 GLY D   1      10.037   1.382  -4.398  0.00 16.57           H
ATOM    123  HA3 GLY D   1       8.837   2.632  -4.014  0.00 16.57           H
ATOM    124  N   ASN D   2       7.656   0.490  -3.155  1.00 15.02           N
ATOM    125  CA  ASN D   2       6.522  -0.395  -2.831  1.00 14.10           C
ATOM    126  C   ASN D   2       5.241   0.104  -3.427  1.00 13.13           C
ATOM    127  O   ASN D   2       4.978   1.309  -3.426  1.00 11.91           O
ATOM    128  CB  ASN D   2       6.346  -0.552  -1.341  1.00 15.38           C
ATOM    129  CG  ASN D   2       7.584  -1.091  -0.692  1.00 14.08           C
ATOM    130  OD1 ASN D   2       8.025  -2.206  -1.016  1.00 17.46           O
ATOM    131  ND2 ASN D   2       8.204  -0.278   0.169  1.00 11.72           N
ATOM    132  H   ASN D   2       8.117   0.901  -2.343  0.00 15.02           H
ATOM    133  HA  ASN D   2       6.744  -1.376  -3.252  0.00 14.10           H
ATOM    134  HB2 ASN D   2       6.124   0.420  -0.900  0.00 15.38           H
ATOM    135  HB3 ASN D   2       5.528  -1.245  -1.146  0.00 15.38           H
ATOM    136 HD21 ASN D   2       9.052  -0.588   0.643  0.00 11.72           H
ATOM    137 HD22 ASN D   2       7.829   0.653   0.352  0.00 11.72           H
TER
ATOM    138  N   ASN E   2       7.656   5.356  -3.155  1.00 15.02           N
ATOM    139  CA  ASN E   2       6.522   4.471  -2.831  1.00 14.10           C
ATOM    140  C   ASN E   2       5.241   4.970  -3.427  1.00 13.13           C
ATOM    141  O   ASN E   2       4.978   6.175  -3.426  1.00 11.91           O
ATOM    142  CB  ASN E   2       6.346   4.314  -1.341  1.00 15.38           C
ATOM    143  CG  ASN E   2       7.584   3.775  -0.692  1.00 14.08           C
ATOM    144  OD1 ASN E   2       8.025   2.660  -1.016  1.00 17.46           O
ATOM    145  ND2 ASN E   2       8.204   4.588   0.169  1.00 11.72           N
ATOM    146  H   ASN E   2       8.117   5.767  -2.343  0.00 15.02           H
ATOM    147  HA  ASN E   2       6.744   3.490  -3.252  0.00 14.10           H
ATOM    148  HB2 ASN E   2       6.124   5.286  -0.900  0.00 15.38           H
ATOM    149  HB3 ASN E   2       5.528   3.621  -1.146  0.00 15.38           H
ATOM    150 HD21 ASN E   2       9.052   4.278   0.643  0.00 11.72           H
ATOM    151 HD22 ASN E   2       7.829   5.519   0.352  0.00 11.72           H
TER
"""

pdb_str2 = """\
CRYST1  375.603  375.603  375.603  90.00  90.00  90.00 P 1
SCALE1      0.002662  0.000000  0.000000        0.00000
SCALE2      0.000000  0.002662  0.000000        0.00000
SCALE3      0.000000  0.000000  0.002662        0.00000
ATOM      1  N   CYS A 297     227.220 185.092 125.379  1.00999.99           N
ATOM      2  CA  CYS A 297     225.963 185.127 124.647  1.00999.99           C
ATOM      3  C   CYS A 297     224.877 185.625 125.592  1.00999.99           C
ATOM      4  O   CYS A 297     224.219 184.840 126.270  1.00999.99           O
ATOM      5  CB  CYS A 297     225.615 183.730 124.139  1.00999.99           C
ATOM      6  SG  CYS A 297     226.811 183.022 122.967  1.00999.99           S
ATOM      7  N   GLU A 298     224.716 186.916 125.639  1.00648.62           N
ATOM      8  CA  GLU A 298     223.728 187.536 126.517  1.00645.56           C
ATOM      9  C   GLU A 298     222.295 187.114 126.232  1.00645.40           C
ATOM     10  O   GLU A 298     221.459 187.089 127.131  1.00640.46           O
ATOM     11  CB  GLU A 298     223.832 189.061 126.422  1.00642.27           C
ATOM     12  CG  GLU A 298     225.230 189.619 126.682  1.00643.47           C
ATOM     13  CD  GLU A 298     225.697 189.398 128.108  1.00645.41           C
ATOM     14  OE1 GLU A 298     226.840 189.793 128.419  1.00643.53           O
ATOM     15  OE2 GLU A 298     224.924 188.841 128.918  1.00646.12           O
ATOM     16  N   GLU A 299     222.008 186.787 124.984  1.00585.91           N
ATOM     17  CA  GLU A 299     220.652 186.396 124.616  1.00583.20           C
ATOM     18  C   GLU A 299     220.337 184.916 124.811  1.00583.15           C
ATOM     19  O   GLU A 299     219.196 184.502 124.639  1.00579.28           O
ATOM     20  CB  GLU A 299     220.373 186.785 123.164  1.00582.77           C
ATOM     21  CG  GLU A 299     221.129 185.959 122.138  1.00585.70           C
ATOM     22  CD  GLU A 299     220.873 186.420 120.713  1.00582.39           C
ATOM     23  OE1 GLU A 299     219.693 186.515 120.327  1.00578.25           O
ATOM     24  OE2 GLU A 299     221.853 186.684 119.987  1.00581.99           O
ATOM     25  N   CYS A 300     221.328 184.123 125.161  1.00527.54           N
ATOM     26  CA  CYS A 300     221.094 182.699 125.369  1.00525.47           C
ATOM     27  C   CYS A 300     220.292 182.507 126.653  1.00524.01           C
ATOM     28  O   CYS A 300     220.523 183.185 127.650  1.00523.47           O
ATOM     29  CB  CYS A 300     222.418 181.945 125.481  1.00525.96           C
ATOM     30  SG  CYS A 300     223.422 181.883 123.976  1.00524.97           S
ATOM     31  N   SER A 301     219.342 181.573 126.612  1.00379.92           N
ATOM     32  CA  SER A 301     218.528 181.276 127.781  1.00380.07           C
ATOM     33  C   SER A 301     219.235 180.206 128.603  1.00379.21           C
ATOM     34  O   SER A 301     220.285 179.702 128.210  1.00380.71           O
ATOM     35  CB  SER A 301     217.144 180.773 127.363  1.00384.83           C
ATOM     36  OG  SER A 301     216.422 181.791 126.707  1.00386.80           O
ATOM     37  N   ILE A 302     218.715 179.908 129.795  1.00581.17           N
ATOM     38  CA  ILE A 302     219.250 178.884 130.669  1.00583.33           C
ATOM     39  C   ILE A 302     219.564 177.578 129.951  1.00585.23           C
ATOM     40  O   ILE A 302     220.659 177.049 130.091  1.00584.97           O
ATOM     41  CB  ILE A 302     218.296 178.592 131.846  1.00582.75           C
ATOM     42  CG1 ILE A 302     218.591 177.219 132.447  1.00582.73           C
ATOM     43  CG2 ILE A 302     216.839 178.644 131.419  1.00578.32           C
ATOM     44  CD1 ILE A 302     217.920 176.995 133.782  1.00582.08           C
ATOM     45  N   ASP A 303     218.629 177.059 129.165  1.00493.36           N
ATOM     46  CA  ASP A 303     218.872 175.809 128.456  1.00493.27           C
ATOM     47  C   ASP A 303     219.956 175.979 127.410  1.00492.13           C
ATOM     48  O   ASP A 303     220.802 175.106 127.219  1.00490.32           O
ATOM     49  CB  ASP A 303     217.598 175.326 127.772  1.00492.03           C
ATOM     50  CG  ASP A 303     216.746 174.485 128.677  1.00491.47           C
ATOM     51  OD1 ASP A 303     216.867 174.632 129.911  1.00492.91           O
ATOM     52  OD2 ASP A 303     215.942 173.680 128.162  1.00486.37           O
ATOM     53  N   GLU A 304     219.919 177.117 126.732  1.00365.23           N
ATOM     54  CA  GLU A 304     220.871 177.408 125.674  1.00365.48           C
ATOM     55  C   GLU A 304     222.301 177.653 126.173  1.00365.04           C
ATOM     56  O   GLU A 304     222.501 178.158 127.278  1.00364.59           O
ATOM     57  CB  GLU A 304     220.389 178.601 124.865  1.00365.54           C
ATOM     58  CG  GLU A 304     218.935 178.513 124.429  1.00366.36           C
ATOM     59  CD  GLU A 304     218.500 179.703 123.596  1.00366.87           C
ATOM     60  OE1 GLU A 304     219.284 180.669 123.482  1.00365.33           O
ATOM     61  OE2 GLU A 304     217.374 179.672 123.056  1.00365.00           O
ATOM     62  N   ASP A 305     223.290 177.295 125.354  1.00691.19           N
ATOM     63  CA  ASP A 305     224.694 177.477 125.703  1.00690.89           C
ATOM     64  C   ASP A 305     225.365 178.422 124.706  1.00691.37           C
ATOM     65  O   ASP A 305     224.938 178.527 123.556  1.00693.54           O
ATOM     66  CB  ASP A 305     225.426 176.133 125.701  1.00690.77           C
ATOM     67  CG  ASP A 305     225.039 175.252 126.874  1.00689.21           C
ATOM     68  OD1 ASP A 305     224.326 175.734 127.778  1.00688.73           O
ATOM     69  OD2 ASP A 305     225.455 174.075 126.893  1.00688.92           O
ATOM     70  N   CYS A 306     226.416 179.106 125.149  1.00999.99           N
ATOM     71  CA  CYS A 306     227.140 180.035 124.307  1.00999.99           C
ATOM     72  C   CYS A 306     228.259 179.313 123.540  1.00999.99           C
ATOM     73  O   CYS A 306     229.299 178.997 124.116  1.00999.99           O
ATOM     74  CB  CYS A 306     227.756 181.153 125.144  1.00999.99           C
ATOM     75  SG  CYS A 306     228.389 182.520 124.142  1.00999.99           S
TER
ATOM      1  N   CYS B 297     227.220 185.092 225.379  1.00999.99           N
ATOM      2  CA  CYS B 297     225.963 185.127 224.647  1.00999.99           C
ATOM      3  C   CYS B 297     224.877 185.625 225.592  1.00999.99           C
ATOM      4  O   CYS B 297     224.219 184.840 226.270  1.00999.99           O
ATOM      5  CB  CYS B 297     225.615 183.730 224.139  1.00999.99           C
ATOM      6  SG  CYS B 297     226.811 183.022 222.967  1.00999.99           S
ATOM      7  N   GLU B 298     224.716 186.916 225.639  1.00648.62           N
ATOM      8  CA  GLU B 298     223.728 187.536 226.517  1.00645.56           C
ATOM      9  C   GLU B 298     222.295 187.114 226.232  1.00645.40           C
ATOM     10  O   GLU B 298     221.459 187.089 227.131  1.00640.46           O
ATOM     11  CB  GLU B 298     223.832 189.061 226.422  1.00642.27           C
ATOM     12  CG  GLU B 298     225.230 189.619 226.682  1.00643.47           C
ATOM     13  CD  GLU B 298     225.697 189.398 228.108  1.00645.41           C
ATOM     14  OE1 GLU B 298     226.840 189.793 228.419  1.00643.53           O
ATOM     15  OE2 GLU B 298     224.924 188.841 228.918  1.00646.12           O
ATOM     16  N   GLU B 299     222.008 186.787 224.984  1.00585.91           N
ATOM     17  CA  GLU B 299     220.652 186.396 224.616  1.00583.20           C
ATOM     18  C   GLU B 299     220.337 184.916 224.811  1.00583.15           C
ATOM     19  O   GLU B 299     219.196 184.502 224.639  1.00579.28           O
ATOM     20  CB  GLU B 299     220.373 186.785 223.164  1.00582.77           C
ATOM     21  CG  GLU B 299     221.129 185.959 222.138  1.00585.70           C
ATOM     22  CD  GLU B 299     220.873 186.420 220.713  1.00582.39           C
ATOM     23  OE1 GLU B 299     219.693 186.515 220.327  1.00578.25           O
ATOM     24  OE2 GLU B 299     221.853 186.684 219.987  1.00581.99           O
ATOM     25  N   CYS B 300     221.328 184.123 225.161  1.00527.54           N
ATOM     26  CA  CYS B 300     221.094 182.699 225.369  1.00525.47           C
ATOM     27  C   CYS B 300     220.292 182.507 226.653  1.00524.01           C
ATOM     28  O   CYS B 300     220.523 183.185 227.650  1.00523.47           O
ATOM     29  CB  CYS B 300     222.418 181.945 225.481  1.00525.96           C
ATOM     30  SG  CYS B 300     223.422 181.883 223.976  1.00524.97           S
ATOM     31  N   SER B 301     219.342 181.573 226.612  1.00379.92           N
ATOM     32  CA  SER B 301     218.528 181.276 227.781  1.00380.07           C
ATOM     33  C   SER B 301     219.235 180.206 228.603  1.00379.21           C
ATOM     34  O   SER B 301     220.285 179.702 228.210  1.00380.71           O
ATOM     35  CB  SER B 301     217.144 180.773 227.363  1.00384.83           C
ATOM     36  OG  SER B 301     216.422 181.791 226.707  1.00386.80           O
ATOM     37  N   ILE B 302     218.715 179.908 229.795  1.00581.17           N
ATOM     38  CA  ILE B 302     219.250 178.884 230.669  1.00583.33           C
ATOM     39  C   ILE B 302     219.564 177.578 229.951  1.00585.23           C
ATOM     40  O   ILE B 302     220.659 177.049 230.091  1.00584.97           O
ATOM     41  CB  ILE B 302     218.296 178.592 231.846  1.00582.75           C
ATOM     42  CG1 ILE B 302     218.591 177.219 232.447  1.00582.73           C
ATOM     43  CG2 ILE B 302     216.839 178.644 231.419  1.00578.32           C
ATOM     44  CD1 ILE B 302     217.920 176.995 233.782  1.00582.08           C
ATOM     45  N   ASP B 303     218.629 177.059 229.165  1.00493.36           N
ATOM     46  CA  ASP B 303     218.872 175.809 228.456  1.00493.27           C
ATOM     47  C   ASP B 303     219.956 175.979 227.410  1.00492.13           C
ATOM     48  O   ASP B 303     220.802 175.106 227.219  1.00490.32           O
ATOM     49  CB  ASP B 303     217.598 175.326 227.772  1.00492.03           C
ATOM     50  CG  ASP B 303     216.746 174.485 228.677  1.00491.47           C
ATOM     51  OD1 ASP B 303     216.867 174.632 229.911  1.00492.91           O
ATOM     52  OD2 ASP B 303     215.942 173.680 228.162  1.00486.37           O
ATOM     53  N   GLU B 304     219.919 177.117 226.732  1.00365.23           N
ATOM     54  CA  GLU B 304     220.871 177.408 225.674  1.00365.48           C
ATOM     55  C   GLU B 304     222.301 177.653 226.173  1.00365.04           C
ATOM     56  O   GLU B 304     222.501 178.158 227.278  1.00364.59           O
ATOM     57  CB  GLU B 304     220.389 178.601 224.865  1.00365.54           C
ATOM     58  CG  GLU B 304     218.935 178.513 224.429  1.00366.36           C
ATOM     59  CD  GLU B 304     218.500 179.703 223.596  1.00366.87           C
ATOM     60  OE1 GLU B 304     219.284 180.669 223.482  1.00365.33           O
ATOM     61  OE2 GLU B 304     217.374 179.672 223.056  1.00365.00           O
ATOM     62  N   ASP B 305     223.290 177.295 225.354  1.00691.19           N
ATOM     63  CA  ASP B 305     224.694 177.477 225.703  1.00690.89           C
ATOM     64  C   ASP B 305     225.365 178.422 224.706  1.00691.37           C
ATOM     65  O   ASP B 305     224.938 178.527 223.556  1.00693.54           O
ATOM     66  CB  ASP B 305     225.426 176.133 225.701  1.00690.77           C
ATOM     67  CG  ASP B 305     225.039 175.252 226.874  1.00689.21           C
ATOM     68  OD1 ASP B 305     224.326 175.734 227.778  1.00688.73           O
ATOM     69  OD2 ASP B 305     225.455 174.075 226.893  1.00688.92           O
ATOM     70  N   CYS B 306     226.416 179.106 225.149  1.00999.99           N
ATOM     71  CA  CYS B 306     227.140 180.035 224.307  1.00999.99           C
ATOM     72  C   CYS B 306     228.259 179.313 223.540  1.00999.99           C
ATOM     73  O   CYS B 306     229.299 178.997 224.116  1.00999.99           O
ATOM     74  CB  CYS B 306     227.756 181.153 225.144  1.00999.99           C
ATOM     75  SG  CYS B 306     228.389 182.520 224.142  1.00999.99           S
TER
ATOM     76  N   CYS K 297     148.993 236.016 133.302  1.00999.99           N
ATOM     77  CA  CYS K 297     150.246 236.048 134.041  1.00999.99           C
ATOM     78  C   CYS K 297     151.341 236.533 133.099  1.00999.99           C
ATOM     79  O   CYS K 297     151.997 235.740 132.429  1.00999.99           O
ATOM     80  CB  CYS K 297     150.583 234.652 134.559  1.00999.99           C
ATOM     81  SG  CYS K 297     149.376 233.958 135.729  1.00999.99           S
ATOM     82  N   GLU K 298     151.510 237.823 133.045  1.00648.62           N
ATOM     83  CA  GLU K 298     152.507 238.431 132.169  1.00645.56           C
ATOM     84  C   GLU K 298     153.936 238.002 132.464  1.00645.40           C
ATOM     85  O   GLU K 298     154.776 237.966 131.570  1.00640.46           O
ATOM     86  CB  GLU K 298     152.412 239.957 132.254  1.00642.27           C
ATOM     87  CG  GLU K 298     151.019 240.522 131.982  1.00643.47           C
ATOM     88  CD  GLU K 298     150.559 240.295 130.555  1.00645.41           C
ATOM     89  OE1 GLU K 298     149.420 240.696 130.235  1.00643.53           O
ATOM     90  OE2 GLU K 298     151.332 239.729 129.753  1.00646.12           O
ATOM     91  N   GLU K 299     154.214 237.680 133.716  1.00585.91           N
ATOM     92  CA  GLU K 299     155.565 237.283 134.094  1.00583.20           C
ATOM     93  C   GLU K 299     155.872 235.800 133.909  1.00583.15           C
ATOM     94  O   GLU K 299     157.009 235.380 134.090  1.00579.28           O
ATOM     95  CB  GLU K 299     155.838 237.679 135.545  1.00582.77           C
ATOM     96  CG  GLU K 299     155.072 236.864 136.572  1.00585.70           C
ATOM     97  CD  GLU K 299     155.323 237.333 137.995  1.00582.39           C
ATOM     98  OE1 GLU K 299     156.501 237.423 138.387  1.00578.25           O
ATOM     99  OE2 GLU K 299     154.340 237.607 138.714  1.00581.99           O
ATOM    100  N   CYS K 300     154.878 235.011 133.559  1.00527.54           N
ATOM    101  CA  CYS K 300     155.104 233.584 133.361  1.00525.47           C
ATOM    102  C   CYS K 300     155.912 233.380 132.082  1.00524.01           C
ATOM    103  O   CYS K 300     155.690 234.053 131.080  1.00523.47           O
ATOM    104  CB  CYS K 300     153.776 232.838 133.246  1.00525.96           C
ATOM    105  SG  CYS K 300     152.763 232.792 134.746  1.00524.97           S
ATOM    106  N   SER K 301     156.855 232.440 132.134  1.00379.92           N
ATOM    107  CA  SER K 301     157.674 232.130 130.972  1.00380.07           C
ATOM    108  C   SER K 301     156.965 231.060 130.152  1.00379.21           C
ATOM    109  O   SER K 301     155.909 230.565 130.543  1.00380.71           O
ATOM    110  CB  SER K 301     159.052 231.621 131.401  1.00384.83           C
ATOM    111  OG  SER K 301     159.777 232.639 132.054  1.00386.80           O
ATOM    112  N   ILE K 302     157.489 230.751 128.965  1.00581.17           N
ATOM    113  CA  ILE K 302     156.953 229.725 128.095  1.00583.33           C
ATOM    114  C   ILE K 302     156.627 228.426 128.819  1.00585.23           C
ATOM    115  O   ILE K 302     155.529 227.903 128.676  1.00584.97           O
ATOM    116  CB  ILE K 302     157.911 229.420 126.925  1.00582.75           C
ATOM    117  CG1 ILE K 302     157.611 228.045 126.330  1.00582.73           C
ATOM    118  CG2 ILE K 302     159.366 229.466 127.359  1.00578.32           C
ATOM    119  CD1 ILE K 302     158.288 227.809 125.001  1.00582.08           C
ATOM    120  N   ASP K 303     157.554 227.906 129.613  1.00493.36           N
ATOM    121  CA  ASP K 303     157.299 226.662 130.328  1.00493.27           C
ATOM    122  C   ASP K 303     156.210 226.845 131.367  1.00492.13           C
ATOM    123  O   ASP K 303     155.358 225.979 131.559  1.00490.32           O
ATOM    124  CB  ASP K 303     158.566 226.175 131.022  1.00492.03           C
ATOM    125  CG  ASP K 303     159.418 225.323 130.127  1.00491.47           C
ATOM    126  OD1 ASP K 303     159.304 225.463 128.892  1.00492.91           O
ATOM    127  OD2 ASP K 303     160.214 224.516 130.652  1.00486.37           O
ATOM    128  N   GLU K 304     156.251 227.987 132.038  1.00365.23           N
ATOM    129  CA  GLU K 304     155.295 228.290 133.089  1.00365.48           C
ATOM    130  C   GLU K 304     153.869 228.541 132.581  1.00365.04           C
ATOM    131  O   GLU K 304     153.678 229.041 131.472  1.00364.59           O
ATOM    132  CB  GLU K 304     155.780 229.485 133.894  1.00365.54           C
ATOM    133  CG  GLU K 304     157.231 229.391 134.338  1.00366.36           C
ATOM    134  CD  GLU K 304     157.669 230.583 135.166  1.00366.87           C
ATOM    135  OE1 GLU K 304     156.890 231.555 135.270  1.00365.33           O
ATOM    136  OE2 GLU K 304     158.792 230.548 135.713  1.00365.00           O
ATOM    137  N   ASP K 305     152.873 228.195 133.397  1.00691.19           N
ATOM    138  CA  ASP K 305     151.473 228.383 133.039  1.00690.89           C
ATOM    139  C   ASP K 305     150.802 229.339 134.026  1.00691.37           C
ATOM    140  O   ASP K 305     151.223 229.448 135.178  1.00693.54           O
ATOM    141  CB  ASP K 305     150.732 227.044 133.045  1.00690.77           C
ATOM    142  CG  ASP K 305     151.120 226.153 131.879  1.00689.21           C
ATOM    143  OD1 ASP K 305     151.841 226.625 130.976  1.00688.73           O
ATOM    144  OD2 ASP K 305     150.697 224.979 131.865  1.00688.92           O
ATOM    145  N   CYS K 306     149.758 230.027 133.573  1.00999.99           N
ATOM    146  CA  CYS K 306     149.035 230.965 134.405  1.00999.99           C
ATOM    147  C   CYS K 306     147.907 230.255 135.170  1.00999.99           C
ATOM    148  O   CYS K 306     146.869 229.942 134.591  1.00999.99           O
ATOM    149  CB  CYS K 306     148.431 232.082 133.558  1.00999.99           C
ATOM    150  SG  CYS K 306     147.801 233.459 134.548  1.00999.99           S
TER
END
"""

def get_geometry_stats(lines, use_ncs=True):
  log_str=StringIO()
  pdb_inp = iotbx.pdb.input(source_info=None, lines=lines)
  m = mmtbx.model.manager(model_input = pdb_inp, log = log_str)
  p = m.get_default_pdb_interpretation_params()
  p.pdb_interpretation.use_ncs_to_build_restraints = use_ncs
  m.process(make_restraints=True, pdb_interpretation_params=p)
  geom=StringIO()
  g = m.geometry_statistics()
  g.show(log=geom, exclude_protein_only_stats=True)
  return geom.getvalue(), log_str.getvalue()

def exercise_01():
  geom_ncs, log_ncs=get_geometry_stats(pdb_str1, True)
  geom_no_ncs, log_no_ncs=get_geometry_stats(pdb_str1, False)
  assert not show_diff(geom_ncs, geom_no_ncs)
  # assert not show_diff(log_ncs, log_no_ncs)
  assert_lines_in_text(log_ncs, """ Restraints were copied for chains:
    B, C
""")
  # print(log_ncs)

def exercise_02():
  geom_ncs, log_ncs=get_geometry_stats(pdb_str2, True)
  # print(geom_ncs)
  # print(log_ncs)
  geom_no_ncs, log_no_ncs=get_geometry_stats(pdb_str2, False)
  assert not show_diff(geom_ncs, geom_no_ncs)
  assert_lines_in_text(log_ncs, """\
    Simple disulfide: pdb=" SG  CYS A 297 " - pdb=" SG  CYS A 306 " distance=2.03
    Simple disulfide: pdb=" SG  CYS B 297 " - pdb=" SG  CYS B 306 " distance=2.03
    Simple disulfide: pdb=" SG  CYS K 297 " - pdb=" SG  CYS K 306 " distance=2.03
""")
  # print(log_ncs)

if(__name__ == "__main__"):
  if libtbx.env.find_in_repositories(relative_path="chem_data") is None:
    print("Skipping exercise_01(): chem_data directory not available")
  else:
    exercise_01()
    exercise_02()
    print('OK')
